#include <stdio.h>
#include "standard.h"
#include "gcrmap.h"
#include "gcrrevmap.h"

/************************************************************************
 * NAME:	a2_gcr4x4_map()   a2_gcr4x4_revmap()
 *
 * DESCR:	Given a byte, map it into 4x4 encoding (2 bytes)   OR
 *		given 2 bytes, reverse map it into a single byte.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
a2_gcr4x4_map(unsigned char in,
	      unsigned char *out1,
	      unsigned char *out2)
{
    *out1 = 0xaa | (in >> 1);
    *out2 = 0xaa | in;
}

unsigned char
a2_gcr4x4_revmap(unsigned char in1,
		 unsigned char in2)
{
    unsigned char out;

    out  = (in1 & 0x55) << 1;
    out |= (in2 & 0x55);

    return(out);
}

/************************************************************************
 * NAME:	6502 Instructions
 *
 * DESCR:	These defines are used to model the Apple ROM code which
 *		is used for the 6x2 and 5x3 encoding.  It was just easy
 *		to take the ROM code and make it work in C in this way.
 *
 ************************************************************************/
#define LSR(arg)	c = (arg) & 0x01;  (arg) = ((arg)>>1) & 0x7f
#define LDA(arg)	a = (arg)
#define ROL(arg)	(arg) = ((arg) << 1) | c
#define STA(arg)	(arg) = a

#define REGISTERS	int a, c, x, y

/************************************************************************
 * NAME:	a2_6x2_encode_aux()
 *
 * DESCR:	Given an incoming buffer of 256 bytes, create the "aux"
 *		buffer of extra data made by swizzling the bits into a
 *		6-bit format.  
 *
 *		Note that this was taken from the Apple ROMs.  In fact,
 *		the behavior of the roms is modeled so I won't introduce
 *		any bugs.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- taken from the ROM code in a convoluted way.
 ************************************************************************/

#define BUF_PNTR	incoming
#define RWTS_BUFFER_2	outgoing
#define RWTS_BUFFER_1	(outgoing+86)

void
a2_6x2_encode_aux(unsigned char *incoming,
		  unsigned char *outgoing)
{
    REGISTERS;

    for (y=2; y!= 0; ) {

	for (x=0; x < 86; x++) {

	    y--;  if (y==-1) y=255;		/* simulates the DEY	*/

	    LDA(BUF_PNTR[y]);

	    LSR(a);
	    ROL(RWTS_BUFFER_2[x]);

	    LSR(a);
	    ROL(RWTS_BUFFER_2[x]);

	    STA(RWTS_BUFFER_1[y]);
	}
    }

    for (x=85; x > -1; x--) {
	RWTS_BUFFER_2[x] &= 0x3f;
    }
}


/************************************************************************
 * NAME:	a2_6x2_decode_aux()
 *
 * DESCR:	Reverse the process of encoding the aux buffer.  Note that
 *		like its opposite, this routine used the ROM code as its
 *		model.  Note, however, that the locations of the buffers
 *		is different, so they are redefined below.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
#undef RWTS_BUFFER_1
#undef RWTS_BUFFER_2
#undef BUF_PNTR

#define RWTS_BUFFER_1	(incoming+86)
#define RWTS_BUFFER_2	incoming
#define BUF_PNTR	outgoing

void
a2_6x2_decode_aux(unsigned char *incoming,	/* assumed to be 342 bytes	*/
		  unsigned char *outgoing)	/* assumed to be 256 bytes	*/
{
    REGISTERS;

    x = 86;

    for (y=0; y < 256; y++) {
	x--;  if (x<0) x=85;

	LDA(RWTS_BUFFER_1[y]);

	LSR(RWTS_BUFFER_2[x]);
	ROL(a);

	LSR(RWTS_BUFFER_2[x]);
	ROL(a);

	STA(BUF_PNTR[y]);
    }
	
}

    
/************************************************************************
 * NAME:	a2_6x2_xorcs()
 *
 * DESCR:	Manipulates the incoming buffer (of 342 bytes) taking the
 *		byte-wise xor, moving the results into the second buffer.
 *		The checksum is returned.
 *
 *		Just like with the rest of the data (eventually) the 
 *		checksum needs to be encoded in 6x2.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
unsigned char 
a2_6x2_xorcs(unsigned char *incoming,		/* assumed to be 342 bytes */
	     unsigned char *outgoing)		/* assumed to be 342 bytes */
{
    int			i;
    unsigned char	a = 0;

    for (i=0; i < 342; i++) {
	a = a ^ incoming[i];
	outgoing[i] = a;
    }

    return(a);
}

a2_6x2_revxorcs(unsigned char *incoming,	/* assumed to be 342 bytes */
		unsigned char *outgoing)	/* assumed to be 342 bytes */
{
    int			i;
    unsigned char	a = 0;

    outgoing[0] = a ^ incoming[0];

    for (i=1; i < 342; i++) {
	a = incoming[i-1] ^ incoming[i];
	outgoing[i] = a;
    }

    return(incoming[i-1]);
}

/************************************************************************
 * NAME:	a2_6x2_reorder()
 *
 * DESCR:	Since the data in the auxillary buffer is ordered from
 *		0 to 85, and it needs to be presented from 85 to 0,
 *		this routine will just reorder it.  It can be used for
 *		both ways, by-the-way.  That is, it undoes itself.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
a2_6x2_reorder(unsigned char *incoming,		/* assumed to be 342 bytes */
	       unsigned char *outgoing)		/* assumed to be 342 bytes */
{
    int	i;

    for (i=0; i < 86; i++) {
	outgoing[85-i] = incoming[i];
    }

    for ( ; i < (256 + 86); i++) {
	outgoing[i] = incoming[i];
    }
}

/************************************************************************
 * NAME:	a2_6x2_aux_pack()
 *
 * DESCR:	Assuming a 342-byte incoming buffer, pack the bits back
 *		into 256 bytes by shifting the bits together in the order
 *		that will be useful by the SVD to generate the 6x2 GCR.
 *
 *		NOTE that 342 bytes of 6 bits of information leads to
 *		2052 bits of info, which is 4 bits longer than 256 bytes
 *		of 8 bits (2048 bits).
 *
 *		SO...the information is packed into 256 bytes, with the
 *		first four bits as a "pre-load" which are kept in the
 *		header block.
 *
 * ARGS:	
 *
 * RETURNS:	the "pre-load" is returned - which is right-justified
 *		4 bits.  This is the first 4 bits of data.
 *
 * NOTES:	- incoming data is organized 6 bits right justified (the
 *		  first two bits of all incoming data is ignored)
 ************************************************************************/
unsigned char
a2_6x2_aux_pack(unsigned char *incoming,	/* 342 bytes */
		unsigned char *outgoing)	/* 256 bytes */
{
    int			i,j;
    unsigned char	preload;

    preload = (incoming[0] >> 2) & 0x0f;	/* grab first four bits */

    /* at this point, there are 2 bits left in the first byte XXXXXXGH	*/

    outgoing[0]  = (incoming[0] << 6) & 0xc0;	/* get remaining 2 bits	*/
    outgoing[0] |= (incoming[1] ) & 0x3f;	/* get 6 bits from #2	*/

    /* now we have 255 bytes left to fill with 340 encoded bytes,	*/
    /* this is (* 255 8) bits which equals (* 340 6)			*/
    /* encoding into the output is: AAAAAABB BBBBCCCC CCDDDDDD		*/
    /* so three bytes of output get 4 bytes of input			*/

    for (i=1, j=2; i < 256; i+=3, j+=4) {
	outgoing[i]    = (incoming[j]  <<2) & 0xfc;	/* 6 bits from #1	*/
	outgoing[i]   |= (incoming[j+1]>>4) & 0x03;	/* 2 bits from #2	*/
	outgoing[i+1]  = (incoming[j+1]<<4) & 0xf0;	/* 4 bits from #2	*/
	outgoing[i+1] |= (incoming[j+2]>>2) & 0x0f;	/* 4 bits from #3	*/
	outgoing[i+2]  = (incoming[j+2]<<6) & 0xc0;	/* 2 bits from #3	*/
	outgoing[i+2] |= (incoming[j+3]   ) & 0x3f;	/* 6 bits from #4	*/
    }

    return(preload);
}

/************************************************************************
 * NAME:	a2_6x2_aux_unpack()
 *
 * DESCR:	"Unpack" the bit-swizzled data from the SVD.  See the
 *		a2_6x2_pack() routine for more information.
 *
 * ARGS:	the "preload" is the front-end 4 bits, right justified.
 *
 * RETURNS:	nothing returned
 *
 * NOTES:	
 ************************************************************************/
void
a2_6x2_aux_unpack(unsigned char *incoming,	/* 256 bytes	*/
		  unsigned char *outgoing,	/* 342 bytes	*/
		  unsigned char  preload)
{
    int	i,j;

    outgoing[0]  = (preload     << 2) & 0x3c;	/* four bits from preload */
    outgoing[0] |= (incoming[0] >> 6) & 0x03;	/* two bits from one	*/
    outgoing[1]  = (incoming[0]     ) & 0x3f;	/* six more from one	*/

    for (i=2, j=1; i < 342; i += 4, j += 3) {
	outgoing[i]    = (incoming[j]   >> 2) & 0x3f;	/* six bits from one	*/
	outgoing[i+1]  = (incoming[j]   << 4) & 0x30;	/* two bits from one	*/	
	outgoing[i+1] |= (incoming[j+1] >> 4) & 0x0f;	/* four bits from two	*/
	outgoing[i+2]  = (incoming[j+1] << 2) & 0x3c;	/* four bits from two	*/
	outgoing[i+2] |= (incoming[j+2] >> 6) & 0x03;	/* two bits from three	*/
	outgoing[i+3]  = (incoming[j+2]     ) & 0x3f;	/* six bits from three	*/
    }
}

/************************************************************************
 * NAME:	a2_6x2_gcr_decode()
 *
 * DESCR:	Decode the given buffer from GCR data.  The size of the
 *		buffer is given.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it went OK, FALSE if a bad GCR byte was in the buffer.
 *
 * NOTES:	
 ************************************************************************/
int
a2_6x2_gcr_decode(unsigned char *incoming,
		  unsigned char *outgoing,
		  int		 size)
{
    int	i;

    for (i=0; i < size; i++) {
	if ( (outgoing[i] = gcr6x2revmap[incoming[i]]) == 0xff) {
	    return(FALSE);
	}
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	a2_6x2_gcr_encode()
 *
 * DESCR:	Encode the given buffer into GCR data.  The data is
 *		assumed to be right justified in 6 bits.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
a2_6x2_gcr_encode(unsigned char *incoming,
		  unsigned char *outgoing,
		  int		 size)
{
    int	i;

    for (i=0; i < size; i++) {
	outgoing[i] = gcr6x2map[incoming[i]&0x3f];
    }
}

/************************************************************************
 * NAME:	a2_6x2_compose()
 *
 * DESCR:	Composes 6x2 data from a 256-byte buffer.  This consists
 *		of organizing the auxillary buffer, computing it's "xor"
 *		and reordering it appropriately.
 *
 * ARGS:	
 *
 * RETURNS:	the checksum (pre GCR) is returned
 *
 * NOTES:	- the checksum is returned, although it hasn't been GCR'd
 *		  yet.
 ************************************************************************/
unsigned char
a2_6x2_compose(unsigned char *incoming,		/* 256 byte incoming	*/
	       unsigned char *outgoing)		/* 342 byte outgoing	*/
{
    unsigned char b1[342];
    unsigned char b2[342];

    a2_6x2_encode_aux(incoming,b1);

    a2_6x2_reorder(b1,b2);

    return(a2_6x2_revxorcs(b2,outgoing));
}

/************************************************************************
 * NAME:	a2_6x2_decompose()
 *
 * DESCR:	The reverse of a2_6x2_compose(), this routine takes a 342-
 *		byte buffer and decomposes the data into 256 bytes.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it worked, FALSE indicates a bad checksum
 *
 * NOTES:	- the checksum is used as a comparison
 *		- even if the checksum comparison is wrong, the entire
 *		  operation is carried out...this is useful if you don't
 *		  have the checksum at the time.
 ************************************************************************/
int
a2_6x2_decompose(unsigned char *incoming,
		 unsigned char *outgoing,
		 unsigned char checksum)
{
    unsigned char b1[342];
    unsigned char b2[342];
    unsigned char cs;

    cs = a2_6x2_xorcs(incoming,b1);

    a2_6x2_reorder(b1,b2);

    a2_6x2_decode_aux(b2,outgoing);


    return(cs == checksum);
}

/************************************************************************
 * NAME:	a2_6x2_gcr_translate()  &  a2_6x2_gcr_revtranslate()
 *
 * DESCR:	Translate a given 256-byte raw data buffer into a 342-byte 
 *		gcr6x2 data buffer and vice-versa.
 *
 * ARGS:	
 *
 * RETURNS:	The checksum is returned in _translate.
 *		TRUE of FALSE is returned in _revtranslate.
 *		If FALSE, then either a bad GCR byte was given or the
 *		the checksum was bad.
 *
 * NOTES:	- The checksum is given in GCR too.
 ************************************************************************/
unsigned char
a2_6x2_gcr_translate(unsigned char *incoming,	/* 256 bytes	*/
		     unsigned char *outgoing)	/* 342 bytes	*/
{
    unsigned char cs;
    unsigned char b1[342];

    cs = a2_6x2_compose(incoming,b1);

    a2_6x2_gcr_encode(b1,outgoing,sizeof(b1));
    a2_6x2_gcr_encode(&cs,&cs,1);

    return(cs);
}

int
a2_6x2_gcr_revtranslate(unsigned char *incoming,	/* 342 bytes	*/
			unsigned char *outgoing,	/* 256 bytes	*/
			unsigned char checksum)
{
    unsigned char decoded;
    unsigned char b1[342];

    if (!a2_6x2_gcr_decode(incoming,b1,sizeof(b1))) {
	return(FALSE);
    }

    if (!a2_6x2_gcr_decode(&checksum,&decoded,1)) {
	return(FALSE);
    }

    return(a2_6x2_decompose(b1,outgoing,decoded));
}

/************************************************************************
 * NAME:	a2_6x2_pack() & a2_6x2_unpack()
 *
 * DESCR:	Given 256 bytes of data, pack it into 256 bytes of GCR-
 *		ready data for the SVD.  One byte with 4 bits of preload
 *		is returned.  See a2_6x2_aux_pack() for more info on
 *		packing.
 *
 * ARGS:	
 *
 * RETURNS:	the preload byte - 4 bits of right-justified data
 *
 * NOTES:	
 ************************************************************************/
unsigned char
a2_6x2_pack(unsigned char *incoming,		/* 256 bytes */
	    unsigned char *outgoing)		/* 256 bytes */
{
    unsigned char	checksum;
    unsigned char	b1[342];

    checksum = a2_6x2_compose(incoming,b1);

    /* the checksum is ignored in this routine	*/

    return(a2_6x2_aux_pack(b1,outgoing));
}

void
a2_6x2_unpack(unsigned char *incoming,		/* 256 bytes */
	      unsigned char *outgoing,		/* 256 bytes */
	      unsigned char  preload)
{
    unsigned char	b1[342];

    
    a2_6x2_aux_unpack(incoming,b1,preload);

    /* note that the return value of _decompose is ignored	*/

    (void) a2_6x2_decompose(b1,outgoing,0);
}

/************************************************************************
 * NAME:	a2_5x3_prenibblize()
 *
 * DESCR:	This routine was translated from the Apple 2 ROM code
 *		of DOS 3.2.1 from B800 to BCFF.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- this code follows the structure of the commented code
 *		  found in http://www.alfter.us/files/aal/aal.8105.bxy
 *		  which is the Apple Assembly Line Archive.
 *		- the same structure of the original code was followed
 *		  so to limit the number of mistakes I could make! :-)
 *		  Sure it is inefficient and could be dramatically
 *		  simplified...but why?
 *
 * Starting locations for buffers (in decimal to make it obvious) 
 *
 *   BUF11 -	  0
 *   BUF12 -	 51
 *   BUF13 -	102
 *   BUF14 - 	153
 *   BUF15 - 	204
 *   BUF16 -	255
 *
 *   BUF21 -	256
 *   BUF22 -	307
 *   BUF23 -	358
 *   BUF24 -	409
 ************************************************************************/
#define BUF11(b)	(b+0x00)
#define BUF12(b)	(b+0x33)
#define BUF13(b)	(b+0x66)
#define BUF14(b)	(b+0x99)
#define BUF15(b)	(b+0xCC)
#define BUF16(b)	(b+0xFF)

#define BUF21(b)	(b+0x100)
#define BUF22(b)	(b+0x133)
#define BUF23(b)	(b+0x166)
#define BUF24(b)	(b+0x199)

a2_5x3_prenibblize(unsigned char *input, unsigned char *output)
{
    unsigned char	t26, t27, t2A;
    int			i;

    for (i=0; i < 51; i++) {

	/* part 1, section 1	*/

	t26 = *input;
	BUF11(output)[i] = (*input >> 3) & 0x1f;
	input++;

	/* part 1, section 2	*/

	t27 = *input;		
	BUF12(output)[i] = (*input >> 3) & 0x1f;
	input++;

	/* part 1, section 3	*/

	t2A = *input;
	BUF13(output)[i] = (*input >> 3) & 0x1f;
	input++;

	/* part 1, section 4	*/

	t2A = (t2A << 1) | ((*input     ) & 0x01);
	t27 = (t27 << 1) | ((*input >> 1) & 0x01);
	t26 = (t26 << 1) | ((*input >> 2) & 0x01);
	BUF14(output)[i] = (*input >> 3) & 0x1f;
	input++;

	/* part 1, section 5	*/
	
	t2A = (t2A << 1) | ((*input     ) & 0x01);
	t27 = (t27 << 1) | ((*input >> 1) & 0x01);
	t26 = (t26 << 1) | ((*input >> 2) & 0x01);
	BUF15(output)[i] = (*input >> 3) & 0x1f;
	input++;

	/* part 2, section 1	*/

	BUF21(output)[i] = t26 & 0x1f;
	
	/* part 2, section 2	*/

	BUF22(output)[i] = t27 & 0x1f;
	
	/* part 2, section 3	*/

	BUF23(output)[i] = t2A & 0x1f;
    }

    *BUF24(output) = *input & 0x07;	/* three bits of last byte	*/

    *BUF16(output) = (*input >> 3) & 0x1f;
}

/************************************************************************
 * NAME:	a2_5x3_postnibblize()
 *
 * DESCR:	Reverses the nibblizing from 410 bytes down to an original
 *		256.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
a2_5x3_postnibblize(unsigned char *input, unsigned char *output)
{
    unsigned char	t26, t27, t2A;
    int	i;

    for (i=0; i < 51; i++) {
    }
}


/************************************************************************
 * NAME:	a2_5x3_reorder()
 *
 * DESCR:	Places the bytes into the order in which they are read/
 *		written onto the disk.
 *
 * ARGS:	both buffers are 410 bytes
 *
 * RETURNS:	
 *
 * NOTES:	- this code was also inspired from the Archive
 ************************************************************************/
a2_5x3_reorder_outgoing(unsigned char *input, unsigned char *output)
{
    int	i;

    for (i=0; i < 154; i++) {
	*output++ = BUF21(input)[i];
    }

    for (i=0; i < 256; i++) {
	*output++ = BUF11(input)[i];
    }
}

a2_5x3_reorder_incoming(unsigned char *input, unsigned char *output)
{
    int	i;

    for (i=0; i < 154; i++) {
	BUF21(output)[i] = *input++;
    }

    for (i=0; i < 256; i++) {
	BUF11(output)[i] = *input++;
    }
}

a2_5x3_svd_nibblize()
{
}

	
a2_5x3_gcr_translate()
{
}
a2_5x3_gcr_revtranslate()
{
}

a2_5x3_pack()
{
}
a2_5x3_unpack()
{
}
